<?php
session_start();
require __DIR__ . '/config.php';

/* ===== Rotas utilitárias ===== */
if (isset($_GET['logout'])) {
  session_destroy();
  header("Location: inicio");
  exit;
}

if (isset($_GET['forceExit'])) {
  // Sai da consulta sem exibir mensagem na URL
  $_SESSION['flash_erro'] = "Seu caminhão não está mais na fila.";
  unset($_SESSION['placa']);
  header("Location: inicio");
  exit;
}

if (isset($_GET['apiFail'])) {
  $_SESSION['flash_erro'] = "Falha na comunicação com a API. Tente novamente.";
  header("Location: inicio");
  exit;
}

/* ===== Entrada padrão (validação no primeiro acesso) ===== */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $placa = strtoupper(trim($_POST['placa'] ?? ''));

  if ($placa === '') {
    $_SESSION['flash_erro'] = "Informe a placa.";
    header("Location: inicio");
    exit;
  }

  if (!preg_match('/^[A-Z0-9]{7,8}$/', $placa)) {
    $_SESSION['flash_erro'] = "Placa incorreta. Use apenas letras e números (7 ou 8 caracteres).";
    header("Location: inicio");
    exit;
  }

  // ⚠️ Não chama a API aqui para não travar o POST
  $_SESSION['placa'] = $placa;

} else {
  // Acesso direto sem POST: precisa ter placa na sessão
  if (empty($_SESSION['placa'])) {
    header("Location: inicio");
    exit;
  }
}

$placa = $_SESSION['placa'];
?>
<!doctype html>
<html lang="pt-br">
<head>
  <meta charset="utf-8">
  <title>Consulta de Fila</title>
  <meta name="viewport" content="width=device-width, initial-scale=1, viewport-fit=cover">
  <meta name="theme-color" content="#0b2d55">
  <link rel="stylesheet" href="assets/style.css">
  <link rel="stylesheet" href="assets/consulta.css">
  <link rel="icon" type="image/png" href="assets/Logo-ATT2.png">
  <style>
    /* SafeArea/StatusBar para mobile */
    body {
      padding-top: env(safe-area-inset-top);
      padding-bottom: env(safe-area-inset-bottom);
      background-color: #fff;
    }
    .container {
      min-height: calc(100vh - env(safe-area-inset-top) - env(safe-area-inset-bottom));
      display: flex;
      flex-direction: column;
      justify-content: space-between;
    }
  </style>
</head>
<body>
  <div class="container">

    <div>
      <div class="header">
        <a class="btn" href="?logout=1">← Voltar</a>
        <span class="badge wait" id="statusBadge">Carregando...</span>
      </div>

      <div class="titulo" id="tituloCaminhao" style="text-align:center; width:100%;">
        📋 Caminhão <?= htmlspecialchars($placa) ?>
      </div>

      <p class="subtitulo" id="subInfo"></p>
      <div class="info-grid" id="cards"></div>
    </div>

    <div class="footer">© <?= date('Y') ?> Todos os direitos reservados Grupo ATT</div>
  </div>

  <script>
    const cardsEl   = document.getElementById('cards');
    const tituloEl  = document.getElementById('tituloCaminhao');
    const subInfoEl = document.getElementById('subInfo');
    const statusBadge = document.getElementById('statusBadge');

    function hhmm(horaCompleta){
      if(!horaCompleta) return "-";
      const partes = horaCompleta.split(':').slice(0,2);
      return partes.join(':');
    }

    function dataBr(yyyy_mm_dd){
      if(!yyyy_mm_dd) return "-";
      const [y,m,d] = yyyy_mm_dd.split('-');
      return `${d.padStart(2,'0')}/${m.padStart(2,'0')}/${y}`;
    }

    function montarCard(label, valor, alerta=false){
      const div = document.createElement('div');
      div.className = 'card' + (alerta ? ' card-alerta' : '');
      div.innerHTML = `<div class="label"><strong>${label}</strong></div><div class="valor">${valor}</div>`;
      return div;
    }

    async function carregar(){
      try{
        statusBadge.textContent = 'Atualizando...';

        // evita cache
        const r = await fetch('api_consulta.php?_=' + Date.now(), { cache:'no-store' });
        const data = await r.json();

        if (data.erro){
          // API falhou: seta flash no servidor e volta ao index, sem query visível
          window.location.href = "resultado?apiFail=1";
          return;
        }

        const veiculo = (data.dados && data.dados.dadosVeiculo) ? data.dados.dadosVeiculo : null;

        if (!veiculo) {
          // caminhão sumiu da fila: seta flash e volta pro index, sem query visível
          window.location.href = "resultado?forceExit=1";
          return;
        }

        const foiChamado = !!(veiculo.data_chamada_descarga && veiculo.hora_chamada_descarga);

        // Título central, sem status textual
        tituloEl.textContent = `📋 Caminhão ${veiculo.veiculo_fila_descarga}`;

        // Mantém o badge de status
        statusBadge.className = 'badge ' + (foiChamado ? 'success' : 'wait');
        statusBadge.textContent = foiChamado ? 'Chamado' : 'Aguardando';

        const infos = [
          { label: "Sua Posição na Fila", value: veiculo.ordem },
          { label: "Empresa", value: veiculo.nome_empresa },
          { label: "Produto", value: veiculo.produto || "-" },
          { label: "Placa do seu Caminhão", value: veiculo.veiculo_fila_descarga },
          ...(veiculo.placa_anterior ? [{ label: "Placa do caminhão à sua frente", value: veiculo.placa_anterior }] : []),
          { label: "Número do Contrato", value: veiculo.contrato_fila_descarga },
          {
            label: "Horário que entrou na fila",
            value: (veiculo.data_fila_descarga && veiculo.hora_fila_descarga)
              ? `${dataBr(veiculo.data_fila_descarga)} às ${hhmm(veiculo.hora_fila_descarga)}`
              : "-",
          },
          {
            label: "Horário que foi chamado",
            value: (veiculo.data_chamada_descarga && veiculo.hora_chamada_descarga)
              ? `${dataBr(veiculo.data_chamada_descarga)} às ${hhmm(veiculo.hora_chamada_descarga)}`
              : "Aguardando ser chamado",
            alerta: (veiculo.data_chamada_descarga && veiculo.hora_chamada_descarga)
          },
        ];

        cardsEl.innerHTML = '';
        for (const item of infos){
          cardsEl.appendChild(montarCard(item.label, item.value, !!item.alerta && item.label === 'Horário que foi chamado'));
        }

        subInfoEl.textContent = '';
      } catch(e){
        // Em exceção, mesma lógica de falha de API
        window.location.href = "resultado?apiFail=1";
      }
    }

    // primeira carga + atualização automática (4 minutos)
    carregar();
    setInterval(carregar, 240000);
  </script>
</body>
</html>
